# 🔧 RIBW - Entwicklerdokumentation

## 📖 Inhaltsverzeichnis

1. [Systemarchitektur](#systemarchitektur)
2. [Code-Struktur](#code-struktur)
3. [Kernfunktionen](#kernfunktionen)
4. [DAW-Integration](#daw-integration)
5. [Live-Test-Suite](#live-test-suite)
6. [Performance-Optimierung](#performance-optimierung)
7. [API-Referenz](#api-referenz)
8. [Entwicklungsumgebung](#entwicklungsumgebung)

---

## 🏗️ Systemarchitektur

### Überblick

RIBW ist als **Single-File-Web-Application** konzipiert, die alle Funktionalitäten in einer HTML-Datei vereint. Das System basiert auf modernen Web-Standards und nutzt keine externen Frameworks.

### Technologie-Stack

```
Frontend: HTML5 + CSS3 + Vanilla JavaScript
Audio: Web Audio API + Tone.js
Notation: VexFlow
Testing: Custom Test Suite
Build: Single-File (kein Build-Prozess)
```

### Architektur-Prinzipien

- **Modularität**: Jede Funktionalität ist in eigenständige Module gekapselt
- **Event-Driven**: Asynchrone Verarbeitung für bessere Performance
- **Progressive Enhancement**: Grundfunktionen funktionieren ohne JavaScript
- **Offline-First**: Vollständige Funktionalität nach dem ersten Laden

---

## 📁 Code-Struktur

### Hauptstruktur

```html
<!DOCTYPE html>
<html>
<head>
  <!-- Meta-Tags und CSS -->
  <!-- Externe Bibliotheken (Tone.js, VexFlow) -->
</head>
<body>
  <!-- Haupt-UI: Gitarren-Tools -->
  <!-- DAW-Interface (überlagert) -->
  <!-- Live-Test-Panel -->
  
  <!-- JavaScript-Module -->
  <script>
    // 1. Gitarren-Tools (Akkorde, Griffbilder, TAB)
    // 2. DAW-Integration (Waveform-Editor, Effekte)
    // 3. Live-Test-Suite (Hardware-Tests, Performance)
    // 4. UI-Management (Dropdowns, Events)
  </script>
</body>
</html>
```

### Modul-Organisation

#### 1. Gitarren-Tools-Modul
```javascript
// Akkord-Parsing und -Verarbeitung
const chordParser = {
  parseChord(chordString) { /* ... */ },
  transposeChord(chord, semitones) { /* ... */ },
  generateVoicings(chord, tuning) { /* ... */ }
};

// Griffbild-Generierung
const fretboardRenderer = {
  renderChord(chord, voicing) { /* ... */ },
  generateSVG(fretboard) { /* ... */ }
};
```

#### 2. DAW-Modul
```javascript
// Audio-Processing
const audioEngine = {
  init() { /* ... */ },
  loadAudio(file) { /* ... */ },
  applyEffects(audioBuffer) { /* ... */ }
};

// Waveform-Editor
const waveformEditor = {
  renderWaveform(audioBuffer) { /* ... */ },
  handleSelection(start, end) { /* ... */ },
  applyEdit(editType, selection) { /* ... */ }
};
```

#### 3. Test-Suite-Modul
```javascript
// Live-Test-System
const LiveTestSuite = {
  tests: [],
  results: {},
  addTest(name, testFn) { /* ... */ },
  runAll() { /* ... */ },
  generateReport() { /* ... */ }
};
```

---

## ⚙️ Kernfunktionen

### Akkord-Parsing-System

#### Akkord-String-Parser
```javascript
function parseChordString(input) {
  // Eingabe: "C G Am F"
  // Ausgabe: [{root: 'C', quality: 'maj'}, {root: 'G', quality: 'maj'}, ...]
  
  const chords = input.trim().split(/\s+/);
  return chords.map(chord => parseChord(chord));
}

function parseChord(chordString) {
  // Unterstützte Formate:
  // C, Cm, C7, Cmaj7, Cm7, Cdim, Caug, Csus2, Csus4, C/G
  
  const root = chordString.match(/^[A-G]#?b?/)[0];
  const quality = chordString.match(/(?:maj|min|m|7|maj7|m7|dim|aug|sus[24])/);
  const bass = chordString.match(/\/([A-G]#?b?)$/);
  
  return {
    root: root,
    quality: quality ? quality[0] : 'maj',
    bass: bass ? bass[1] : null
  };
}
```

#### Transposition-Engine
```javascript
const transpositionEngine = {
  // Halbton-Offset für jede Note
  noteOffsets: {
    'C': 0, 'C#': 1, 'Db': 1, 'D': 2, 'D#': 3, 'Eb': 3,
    'E': 4, 'F': 5, 'F#': 6, 'Gb': 6, 'G': 7, 'G#': 8,
    'Ab': 8, 'A': 9, 'A#': 10, 'Bb': 10, 'B': 11
  },
  
  transposeChord(chord, semitones) {
    const newRoot = this.transposeNote(chord.root, semitones);
    return { ...chord, root: newRoot };
  },
  
  transposeNote(note, semitones) {
    const offset = this.noteOffsets[note];
    const newOffset = (offset + semitones + 12) % 12;
    return Object.keys(this.noteOffsets).find(key => 
      this.noteOffsets[key] === newOffset
    );
  }
};
```

### Voicing-Generator

#### Fingersatz-Optimierung
```javascript
const voicingGenerator = {
  // Standard-Gitarren-Tuning (E A D G B E)
  standardTuning: ['E2', 'A2', 'D3', 'G3', 'B3', 'E4'],
  
  generateVoicings(chord, tuning = this.standardTuning) {
    const notes = this.getChordNotes(chord);
    const voicings = [];
    
    // Generiere verschiedene Voicing-Positionen
    for (let startFret = 0; startFret <= 12; startFret++) {
      const voicing = this.generateVoicingAtFret(notes, tuning, startFret);
      if (voicing && this.isPlayable(voicing)) {
        voicings.push(voicing);
      }
    }
    
    return this.optimizeVoicings(voicings);
  },
  
  generateVoicingAtFret(notes, tuning, startFret) {
    const voicing = [];
    
    for (let string = 0; string < 6; string++) {
      const openNote = tuning[string];
      const noteAtFret = this.transposeNote(openNote, startFret);
      
      // Finde passende Note aus dem Akkord
      const matchingNote = this.findClosestNote(noteAtFret, notes);
      if (matchingNote) {
        voicing.push({
          string: string,
          fret: startFret + this.getNoteDistance(openNote, matchingNote),
          note: matchingNote
        });
      }
    }
    
    return voicing;
  }
};
```

---

## 🎛️ DAW-Integration

### Audio-Engine-Architektur

#### Web Audio API Integration
```javascript
class AudioEngine {
  constructor() {
    this.audioContext = null;
    this.masterGain = null;
    this.effectsChain = null;
    this.audioBuffer = null;
  }
  
  async init() {
    // AudioContext erstellen
    this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
    
    // Master-Gain-Node
    this.masterGain = this.audioContext.createGain();
    this.masterGain.gain.value = 1.0;
    
    // Effekt-Kette aufbauen
    this.effectsChain = this.createEffectsChain();
    
    // Verbindungen herstellen
    this.effectsChain.input.connect(this.masterGain);
    this.masterGain.connect(this.audioContext.destination);
  }
  
  createEffectsChain() {
    const input = this.audioContext.createGain();
    
    // 3-Band EQ
    const lowShelf = this.audioContext.createBiquadFilter();
    lowShelf.type = 'lowshelf';
    lowShelf.frequency.value = 120;
    lowShelf.gain.value = 0;
    
    const midPeak = this.audioContext.createBiquadFilter();
    midPeak.type = 'peaking';
    midPeak.frequency.value = 1000;
    midPeak.Q.value = 1;
    midPeak.gain.value = 0;
    
    const highShelf = this.audioContext.createBiquadFilter();
    highShelf.type = 'highshelf';
    highShelf.frequency.value = 6000;
    highShelf.gain.value = 0;
    
    // Stereo-Pan
    const stereoPanner = this.audioContext.createStereoPanner();
    stereoPanner.pan.value = 0;
    
    // Verbindungen
    input.connect(lowShelf);
    lowShelf.connect(midPeak);
    midPeak.connect(highShelf);
    highShelf.connect(stereoPanner);
    
    return {
      input,
      lowShelf,
      midPeak,
      highShelf,
      stereoPanner,
      output: stereoPanner
    };
  }
}
```

### Waveform-Editor

#### Canvas-basierte Darstellung
```javascript
class WaveformEditor {
  constructor(canvas, audioEngine) {
    this.canvas = canvas;
    this.ctx = canvas.getContext('2d');
    this.audioEngine = audioEngine;
    this.selection = { start: 0, end: 0 };
    this.isSelecting = false;
    
    this.setupEventListeners();
  }
  
  setupEventListeners() {
    this.canvas.addEventListener('mousedown', this.handleMouseDown.bind(this));
    this.canvas.addEventListener('mousemove', this.handleMouseMove.bind(this));
    this.canvas.addEventListener('mouseup', this.handleMouseUp.bind(this));
    this.canvas.addEventListener('contextmenu', this.handleContextMenu.bind(this));
  }
  
  renderWaveform(audioBuffer) {
    const { width, height } = this.canvas;
    const channelData = audioBuffer.getChannelData(0);
    const samplesPerPixel = Math.ceil(channelData.length / width);
    
    this.ctx.clearRect(0, 0, width, height);
    this.ctx.fillStyle = '#4cc9f0';
    
    for (let x = 0; x < width; x++) {
      const startSample = x * samplesPerPixel;
      const endSample = Math.min(startSample + samplesPerPixel, channelData.length);
      
      let min = 1, max = -1;
      for (let i = startSample; i < endSample; i++) {
        const sample = channelData[i];
        if (sample < min) min = sample;
        if (sample > max) max = sample;
      }
      
      const y1 = (1 + min) * height / 2;
      const y2 = (1 + max) * height / 2;
      
      this.ctx.fillRect(x, y1, 1, y2 - y1);
    }
  }
  
  handleMouseDown(event) {
    const rect = this.canvas.getBoundingClientRect();
    const x = event.clientX - rect.left;
    
    this.isSelecting = true;
    this.selection.start = x;
    this.selection.end = x;
    
    this.renderSelection();
  }
  
  handleMouseMove(event) {
    if (!this.isSelecting) return;
    
    const rect = this.canvas.getBoundingClientRect();
    const x = event.clientX - rect.left;
    
    this.selection.end = x;
    this.renderSelection();
  }
  
  renderSelection() {
    const start = Math.min(this.selection.start, this.selection.end);
    const end = Math.max(this.selection.start, this.selection.end);
    
    // Auswahl-Highlight zeichnen
    this.ctx.fillStyle = 'rgba(76, 201, 240, 0.3)';
    this.ctx.fillRect(start, 0, end - start, this.canvas.height);
  }
}
```

---

## 🧪 Live-Test-Suite

### Test-Framework-Architektur

#### Test-Suite-Kern
```javascript
class LiveTestSuite {
  constructor() {
    this.tests = [];
    this.results = {};
    this.startTime = Date.now();
    this.isRunning = false;
  }
  
  addTest(name, testFunction, category = 'general') {
    this.tests.push({
      name,
      testFunction,
      category,
      id: this.generateTestId()
    });
  }
  
  async runAll() {
    if (this.isRunning) {
      console.warn('Tests laufen bereits...');
      return;
    }
    
    this.isRunning = true;
    console.log('🚀 Starte Live-Test-Suite...');
    
    for (const test of this.tests) {
      try {
        console.log(`\n🧪 Teste: ${test.name}`);
        const result = await test.testFunction();
        
        this.results[test.id] = {
          name: test.name,
          category: test.category,
          success: true,
          result,
          timestamp: Date.now(),
          duration: Date.now() - this.startTime
        };
        
        console.log(`✅ ${test.name}: ERFOLGREICH`, result);
      } catch (error) {
        this.results[test.id] = {
          name: test.name,
          category: test.category,
          success: false,
          error: error.message,
          stack: error.stack,
          timestamp: Date.now(),
          duration: Date.now() - this.startTime
        };
        
        console.error(`❌ ${test.name}: FEHLGESCHLAGEN - ${error.message}`);
      }
    }
    
    this.isRunning = false;
    this.generateReport();
  }
  
  generateReport() {
    const total = this.tests.length;
    const passed = Object.values(this.results).filter(r => r.success).length;
    const failed = total - passed;
    const successRate = (passed / total) * 100;
    
    const report = {
      summary: {
        total,
        passed,
        failed,
        successRate: Math.round(successRate * 100) / 100,
        duration: Date.now() - this.startTime
      },
      results: this.results,
      recommendations: this.generateRecommendations()
    };
    
    console.log('\n📊 LIVE-TEST-REPORT:', report);
    return report;
  }
  
  generateRecommendations() {
    const recommendations = [];
    
    // Performance-basierte Empfehlungen
    const performanceTests = Object.values(this.results)
      .filter(r => r.category === 'performance');
    
    const slowTests = performanceTests
      .filter(r => r.result && r.result.duration > 100);
    
    if (slowTests.length > 0) {
      recommendations.push({
        type: 'performance',
        severity: 'warning',
        message: `${slowTests.length} Performance-Tests zeigen langsame Ausführung`,
        tests: slowTests.map(t => t.name)
      });
    }
    
    // Audio-basierte Empfehlungen
    const audioTests = Object.values(this.results)
      .filter(r => r.category === 'audio');
    
    const failedAudioTests = audioTests
      .filter(r => !r.success);
    
    if (failedAudioTests.length > 0) {
      recommendations.push({
        type: 'audio',
        severity: 'critical',
        message: `${failedAudioTests.length} Audio-Tests fehlgeschlagen - Live-Einsatz nicht empfohlen`,
        tests: failedAudioTests.map(t => t.name)
      });
    }
    
    return recommendations;
  }
}
```

### Test-Kategorien

#### 1. Hardware-Performance-Tests
```javascript
// CPU-Performance-Test
suite.addTest("CPU-Performance", () => {
  const start = performance.now();
  let result = 0;
  
  // Intensiver Mathematik-Test
  for (let i = 0; i < 1000000; i++) {
    result += Math.sin(i * 0.01) * Math.cos(i * 0.02);
  }
  
  const duration = performance.now() - start;
  
  return {
    duration: `${duration.toFixed(2)}ms`,
    result: result.toFixed(6),
    performance: duration < 50 ? 'EXCELLENT' : 
                 duration < 100 ? 'GOOD' : 
                 duration < 200 ? 'ACCEPTABLE' : 'POOR',
    recommendation: duration > 200 ? 
      'CPU ist möglicherweise zu langsam für Live-Einsatz' : 
      'CPU-Performance ausreichend für Live-Einsatz'
  };
}, 'performance');

// Memory-Performance-Test
suite.addTest("Memory-Management", () => {
  const initialMemory = performance.memory?.usedJSHeapSize || 0;
  
  // Memory-intensive Operationen
  const testArrays = [];
  for (let i = 0; i < 1000; i++) {
    testArrays.push(new Float32Array(10000));
  }
  
  const peakMemory = performance.memory?.usedJSHeapSize || 0;
  
  // Cleanup
  testArrays.length = 0;
  
  const finalMemory = performance.memory?.usedJSHeapSize || 0;
  
  return {
    initialMemory: `${(initialMemory / 1024 / 1024).toFixed(2)}MB`,
    peakMemory: `${(peakMemory / 1024 / 1024).toFixed(2)}MB`,
    finalMemory: `${(finalMemory / 1024 / 1024).toFixed(2)}MB`,
    memoryIncrease: `${((peakMemory - initialMemory) / 1024 / 1024).toFixed(2)}MB`,
    cleanupEfficiency: `${((peakMemory - finalMemory) / (peakMemory - initialMemory) * 100).toFixed(1)}%`,
    status: finalMemory <= initialMemory * 1.1 ? 'EXCELLENT' : 'GOOD'
  };
}, 'performance');
```

#### 2. Audio-Performance-Tests
```javascript
// Audio-Latenz-Test
suite.addTest("Audio-Latenz", async () => {
  if (!window.AudioContext) {
    throw new Error('Web Audio API nicht unterstützt');
  }
  
  const audioContext = new AudioContext();
  const startTime = audioContext.currentTime;
  
  // Test-Oszillator erstellen
  const oscillator = audioContext.createOscillator();
  const gainNode = audioContext.createGain();
  
  oscillator.connect(gainNode);
  gainNode.connect(audioContext.destination);
  
  // Sofort starten
  oscillator.start();
  oscillator.stop(audioContext.currentTime + 0.1);
  
  const latency = (audioContext.currentTime - startTime) * 1000;
  
  return {
    latency: `${latency.toFixed(2)}ms`,
    status: latency < 10 ? 'EXCELLENT' : 
            latency < 50 ? 'GOOD' : 
            latency < 100 ? 'ACCEPTABLE' : 'POOR',
    recommendation: latency > 100 ? 
      'Hohe Audio-Latenz - nicht für Live-Einsatz geeignet' : 
      'Audio-Latenz akzeptabel für Live-Einsatz'
  };
}, 'audio');

// Audio-Processing-Performance
suite.addTest("Audio-Processing-Performance", async () => {
  if (!window.AudioContext) {
    throw new Error('Web Audio API nicht unterstützt');
  }
  
  const audioContext = new AudioContext();
  const bufferSize = 4096;
  const testBuffer = audioContext.createBuffer(2, bufferSize, 44100);
  
  // Test-Audio-Daten generieren
  for (let channel = 0; channel < 2; channel++) {
    const channelData = testBuffer.getChannelData(channel);
    for (let i = 0; i < bufferSize; i++) {
      channelData[i] = Math.random() * 2 - 1;
    }
  }
  
  const start = performance.now();
  
  // Effekt-Kette anwenden
  const source = audioContext.createBufferSource();
  const gainNode = audioContext.createGain();
  const filterNode = audioContext.createBiquadFilter();
  
  source.buffer = testBuffer;
  source.connect(gainNode);
  gainNode.connect(filterNode);
  filterNode.connect(audioContext.destination);
  
  source.start();
  source.stop(audioContext.currentTime + 0.1);
  
  const duration = performance.now() - start;
  
  return {
    bufferSize,
    channels: testBuffer.numberOfChannels,
    sampleRate: testBuffer.sampleRate,
    processingTime: `${duration.toFixed(2)}ms`,
    status: duration < 20 ? 'EXCELLENT' : 
            duration < 50 ? 'GOOD' : 
            duration < 100 ? 'ACCEPTABLE' : 'POOR'
  };
}, 'audio');
```

---

## ⚡ Performance-Optimierung

### Memory-Management

#### Garbage Collection-Optimierung
```javascript
class MemoryManager {
  constructor() {
    this.allocatedObjects = new WeakSet();
    this.memoryLimit = 100 * 1024 * 1024; // 100MB
    this.cleanupInterval = 30000; // 30 Sekunden
    
    this.startCleanupTimer();
  }
  
  trackObject(obj) {
    this.allocatedObjects.add(obj);
    return obj;
  }
  
  startCleanupTimer() {
    setInterval(() => {
      this.performCleanup();
    }, this.cleanupInterval);
  }
  
  performCleanup() {
    const currentMemory = performance.memory?.usedJSHeapSize || 0;
    
    if (currentMemory > this.memoryLimit) {
      console.warn('Memory-Limit überschritten, starte Cleanup...');
      
      // Audio-Buffer freigeben
      if (window.audioBuffers) {
        window.audioBuffers.forEach(buffer => {
          if (buffer && typeof buffer.disconnect === 'function') {
            buffer.disconnect();
          }
        });
        window.audioBuffers = [];
      }
      
      // Canvas-Kontexte bereinigen
      const canvases = document.querySelectorAll('canvas');
      canvases.forEach(canvas => {
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
      });
      
      // Event-Listener bereinigen
      this.cleanupEventListeners();
    }
  }
  
  cleanupEventListeners() {
    // Temporäre Event-Listener entfernen
    const tempElements = document.querySelectorAll('[data-temp-listener]');
    tempElements.forEach(element => {
      element.removeEventListener('click', element.tempClickHandler);
      element.removeAttribute('data-temp-listener');
    });
  }
}
```

### Audio-Optimierung

#### Buffer-Pool-System
```javascript
class AudioBufferPool {
  constructor(maxBuffers = 10) {
    this.maxBuffers = maxBuffers;
    this.availableBuffers = [];
    this.usedBuffers = new Set();
  }
  
  getBuffer() {
    if (this.availableBuffers.length > 0) {
      const buffer = this.availableBuffers.pop();
      this.usedBuffers.add(buffer);
      return buffer;
    }
    
    if (this.usedBuffers.size < this.maxBuffers) {
      const buffer = new AudioBuffer();
      this.usedBuffers.add(buffer);
      return buffer;
    }
    
    // Warte auf freien Buffer
    return this.waitForBuffer();
  }
  
  releaseBuffer(buffer) {
    if (this.usedBuffers.has(buffer)) {
      this.usedBuffers.delete(buffer);
      this.availableBuffers.push(buffer);
    }
  }
  
  async waitForBuffer() {
    return new Promise(resolve => {
      const checkInterval = setInterval(() => {
        if (this.availableBuffers.length > 0) {
          clearInterval(checkInterval);
          const buffer = this.availableBuffers.pop();
          this.usedBuffers.add(buffer);
          resolve(buffer);
        }
      }, 10);
    });
  }
}
```

---

## 📚 API-Referenz

### Gitarren-Tools-API

#### ChordParser
```javascript
class ChordParser {
  /**
   * Parst einen Akkord-String in Akkord-Objekte
   * @param {string} input - Eingabe-String (z.B. "C G Am F")
   * @returns {Array<ChordObject>} Array von Akkord-Objekten
   */
  static parseChordString(input) { /* ... */ }
  
  /**
   * Parst einen einzelnen Akkord
   * @param {string} chordString - Akkord-String (z.B. "Cm7")
   * @returns {ChordObject} Akkord-Objekt
   */
  static parseChord(chordString) { /* ... */ }
  
  /**
   * Transponiert einen Akkord um gegebene Halbtöne
   * @param {ChordObject} chord - Zu transponierender Akkord
   * @param {number} semitones - Anzahl Halbtöne
   * @returns {ChordObject} Transponierter Akkord
   */
  static transposeChord(chord, semitones) { /* ... */ }
}

/**
 * @typedef {Object} ChordObject
 * @property {string} root - Grundton (z.B. "C", "F#")
 * @property {string} quality - Akkord-Qualität (z.B. "maj", "min", "7")
 * @property {string|null} bass - Bass-Note (z.B. "G" für C/G)
 */
```

#### FretboardRenderer
```javascript
class FretboardRenderer {
  /**
   * Rendert ein Griffbild für einen Akkord
   * @param {ChordObject} chord - Zu rendernder Akkord
   * @param {Array<FretPosition>} voicing - Fingersatz
   * @param {Object} options - Rendering-Optionen
   * @returns {string} SVG-String des Griffbilds
   */
  static renderChord(chord, voicing, options = {}) { /* ... */ }
  
  /**
   * Generiert ASCII-TAB für einen Akkord
   * @param {ChordObject} chord - Akkord
   * @param {Array<FretPosition>} voicing - Fingersatz
   * @returns {string} ASCII-TAB-String
   */
  static generateAsciiTab(chord, voicing) { /* ... */ }
}

/**
 * @typedef {Object} FretPosition
 * @property {number} string - Saiten-Index (0-5)
 * @property {number} fret - Bund-Nummer (0-12)
 * @property {string} note - Note an dieser Position
 */
```

### DAW-API

#### AudioEngine
```javascript
class AudioEngine {
  /**
   * Initialisiert die Audio-Engine
   * @returns {Promise<void>}
   */
  async init() { /* ... */ }
  
  /**
   * Lädt eine Audio-Datei
   * @param {File} file - Audio-Datei
   * @returns {Promise<AudioBuffer>} Geladener Audio-Buffer
   */
  async loadAudio(file) { /* ... */ }
  
  /**
   * Spielt Audio ab
   * @param {AudioBuffer} buffer - Zu spielender Buffer
   * @param {number} startTime - Start-Zeit
   * @param {number} duration - Dauer
   */
  playAudio(buffer, startTime = 0, duration = null) { /* ... */ }
  
  /**
   * Stoppt alle Audio-Wiedergabe
   */
  stopAll() { /* ... */ }
}
```

#### WaveformEditor
```javascript
class WaveformEditor {
  /**
   * Rendert eine Waveform
   * @param {AudioBuffer} audioBuffer - Audio-Buffer
   */
  renderWaveform(audioBuffer) { /* ... */ }
  
  /**
   * Setzt die Auswahl
   * @param {number} start - Start-Position (Pixel)
   * @param {number} end - End-Position (Pixel)
   */
  setSelection(start, end) { /* ... */ }
  
  /**
   * Wendet eine Bearbeitung an
   * @param {string} editType - Art der Bearbeitung
   * @param {Object} selection - Auswahl-Bereich
   */
  applyEdit(editType, selection) { /* ... */ }
}
```

---

## 🛠️ Entwicklungsumgebung

### Setup

#### 1. Repository klonen
```bash
git clone [repository-url]
cd RIBW/Secondedition
```

#### 2. Entwicklungsserver starten
```bash
# Python 3
python -m http.server 8000

# Node.js
npx http-server

# PHP
php -S localhost:8000
```

#### 3. Browser öffnen
```
http://localhost:8000/Gitarre%20Html.html
```

### Debugging

#### Browser-Entwicklertools
```javascript
// Debug-Modus aktivieren
window.DEBUG_MODE = true;

// Performance-Monitoring
console.time('operation');
// ... Code ...
console.timeEnd('operation');

// Memory-Profiling
console.log('Memory:', performance.memory);
```

#### Live-Test-Suite im Debug-Modus
```javascript
// Erweiterte Test-Informationen
suite.addTest("Debug-Informationen", () => {
  return {
    userAgent: navigator.userAgent,
    platform: navigator.platform,
    language: navigator.language,
    cookieEnabled: navigator.cookieEnabled,
    onLine: navigator.onLine,
    hardwareConcurrency: navigator.hardwareConcurrency,
    deviceMemory: navigator.deviceMemory,
    maxTouchPoints: navigator.maxTouchPoints
  };
}, 'debug');
```

### Testing

#### Unit-Tests
```javascript
// Einzelne Funktionen testen
function testChordParser() {
  const testCases = [
    { input: 'C', expected: { root: 'C', quality: 'maj', bass: null } },
    { input: 'Am', expected: { root: 'A', quality: 'min', bass: null } },
    { input: 'G7', expected: { root: 'G', quality: '7', bass: null } }
  ];
  
  testCases.forEach(testCase => {
    const result = ChordParser.parseChord(testCase.input);
    const success = JSON.stringify(result) === JSON.stringify(testCase.expected);
    
    console.log(`${success ? '✅' : '❌'} ${testCase.input}:`, 
      success ? 'PASS' : `FAIL (expected: ${JSON.stringify(testCase.expected)}, got: ${JSON.stringify(result)})`);
  });
}
```

#### Integration-Tests
```javascript
// Komplette Workflows testen
async function testCompleteWorkflow() {
  try {
    // 1. Akkorde eingeben
    const chords = ChordParser.parseChordString('C G Am F');
    console.log('✅ Akkorde geparst:', chords.length);
    
    // 2. Voicings generieren
    const voicings = chords.map(chord => 
      VoicingGenerator.generateVoicings(chord)
    );
    console.log('✅ Voicings generiert:', voicings.length);
    
    // 3. Audio abspielen
    await AudioEngine.init();
    console.log('✅ Audio-Engine initialisiert');
    
    // 4. Waveform rendern
    const testBuffer = AudioEngine.createTestBuffer();
    WaveformEditor.renderWaveform(testBuffer);
    console.log('✅ Waveform gerendert');
    
    console.log('🎉 Kompletter Workflow erfolgreich!');
  } catch (error) {
    console.error('❌ Workflow fehlgeschlagen:', error);
  }
}
```

---

## 📝 Best Practices

### Code-Qualität

#### 1. Konsistente Namensgebung
```javascript
// ✅ Gut: Klare, beschreibende Namen
const chordParser = new ChordParser();
const fretboardRenderer = new FretboardRenderer();

// ❌ Schlecht: Unklare Abkürzungen
const cp = new ChordParser();
const fr = new FretboardRenderer();
```

#### 2. Error-Handling
```javascript
// ✅ Gut: Umfassendes Error-Handling
async function loadAudio(file) {
  try {
    if (!file) {
      throw new Error('Keine Datei angegeben');
    }
    
    if (!file.type.startsWith('audio/')) {
      throw new Error('Keine gültige Audio-Datei');
    }
    
    const buffer = await decodeAudioData(file);
    return buffer;
  } catch (error) {
    console.error('Fehler beim Laden der Audio-Datei:', error);
    throw error;
  }
}

// ❌ Schlecht: Kein Error-Handling
async function loadAudio(file) {
  const buffer = await decodeAudioData(file);
  return buffer;
}
```

#### 3. Performance-Optimierung
```javascript
// ✅ Gut: Effiziente Schleifen
function processAudioData(channelData) {
  const length = channelData.length;
  for (let i = 0; i < length; i++) {
    channelData[i] *= 0.5; // Gain reduzieren
  }
}

// ❌ Schlecht: Ineffiziente Array-Methoden
function processAudioData(channelData) {
  channelData.forEach((sample, i) => {
    channelData[i] = sample * 0.5;
  });
}
```

### Dokumentation

#### 1. JSDoc-Kommentare
```javascript
/**
 * Transponiert einen Akkord um gegebene Halbtöne
 * @param {ChordObject} chord - Zu transponierender Akkord
 * @param {number} semitones - Anzahl Halbtöne (positiv = hoch, negativ = runter)
 * @returns {ChordObject} Transponierter Akkord
 * @example
 * const original = { root: 'C', quality: 'maj' };
 * const transposed = transposeChord(original, 2); // C -> D
 * console.log(transposed.root); // 'D'
 */
function transposeChord(chord, semitones) {
  // Implementation...
}
```

#### 2. README-Dateien
```markdown
# Modul-Name

Kurze Beschreibung der Funktionalität.

## Verwendung

```javascript
import { ModuleName } from './module-name.js';

const instance = new ModuleName();
instance.doSomething();
```

## API-Referenz

### `doSomething()`

Führt eine Aktion aus.

**Parameter:**
- `param1` (string): Beschreibung des Parameters

**Rückgabewert:**
- (boolean): Erfolg der Operation

## Beispiele

Weitere Verwendungsbeispiele...
```

---

## 🚀 Deployment

### Produktions-Build

#### 1. Code-Minifizierung
```bash
# JavaScript minifizieren
npx terser "Gitarre Html.html" --compress --mangle --output "gitarre.min.html"

# CSS minifizieren
npx clean-css-cli styles.css --output styles.min.css
```

#### 2. Asset-Optimierung
```bash
# Bilder komprimieren
npx imagemin assets/* --out-dir=assets/optimized

# Audio-Dateien konvertieren
ffmpeg -i input.wav -c:a libmp3lame -b:a 128k output.mp3
```

#### 3. Gzip-Kompression
```bash
# HTML komprimieren
gzip -9 "gitarre.min.html"

# CSS komprimieren
gzip -9 "styles.min.css"
```

### CDN-Deployment

#### 1. GitHub Pages
```bash
# Repository auf GitHub hochladen
git add .
git commit -m "Release v2.0"
git push origin main

# GitHub Pages aktivieren in Repository-Einstellungen
```

#### 2. Netlify
```bash
# Netlify CLI installieren
npm install -g netlify-cli

# Deploy
netlify deploy --prod --dir=.
```

---

## 🔮 Zukunftsperspektiven

### Geplante Features

#### 1. MIDI-Integration
- **Hardware-MIDI** Unterstützung
- **MIDI-Dateien** importieren/exportieren
- **MIDI-Controller** Mapping

#### 2. Erweiterte Effekte
- **VST-Plugins** Integration
- **Custom-Effekte** erstellen
- **Effekt-Presets** verwalten

#### 3. Cloud-Integration
- **Projekt-Synchronisation** über Cloud
- **Kollaborative Bearbeitung** in Echtzeit
- **Backup & Versionierung**

#### 4. Mobile-Optimierung
- **Progressive Web App** (PWA)
- **Touch-Gesten** für mobile Geräte
- **Offline-Funktionalität** erweitern

### Performance-Verbesserungen

#### 1. WebAssembly
- **Audio-Processing** in WASM
- **Effekt-Algorithmen** optimieren
- **Real-Time-Performance** steigern

#### 2. Web Workers
- **Background-Processing** für Audio
- **UI-Responsiveness** verbessern
- **Multi-Core-Nutzung** optimieren

#### 3. Audio Worklets
- **Custom-Audio-Nodes** implementieren
- **Low-Latency-Processing** ermöglichen
- **Real-Time-Effekte** verbessern

---

**🎵 Viel Erfolg bei der Entwicklung mit RIBW! 🎵**

*Bei technischen Fragen: Code studieren, Tests ausführen, Community konsultieren.*

---

*Letzte Aktualisierung: Dezember 2024*
*Version: 2.0 - Vollständiges System*
*Sprache: Deutsch*
*Zielgruppe: Entwickler*
